Rem 
Rem ordcwmig.sql
Rem 
Rem Copyright (c) 2005, 2009, Oracle and/or its affiliates. 
Rem All rights reserved. 
Rem 
Rem  NAME
Rem    ordplscw.sql - Oracle Multimedia Code Wizard migration utility
Rem 
Rem  DESCRIPTION
Rem   In pre 10.2 versions of the Code Wizard, the username for the ORDCWADMIN 
Rem   DAD was hard-coded to ORDSYS and authorized DADs were stored in the
Rem   ORDSYS schema.  ORDSYS is now a locked account, and this old scheme
Rem   will only work if ORDSYS is unlocked.
Rem
Rem   In this version, the user for the ORDCWADMIN DAD is specified when the
Rem   Code Wizard is installed. This package migrates the authorized 
Rem   DADs stored in ORDSYS.OrdPlsGwyAuthedDADs to
Rem   <admin_user_name>.OrdPlsGwyAuthedDADs
Rem
Rem   This package is installed in the <admin_user_name> schema with 
Rem   INVOKERs rights. No PUBLIC execute privs are granted on this package. 
Rem   Thus you must be SYS or <admin_user_name> to execute. 
Rem   The package is dropped after the migration is performed.
Rem
Rem   NOTES
Rem    This script must be invoked from the installer ordplsci.sql.
Rem    See the Oracle Multimedia Code Wizard README document for more 
Rem    information.
Rem 


---------------------------------------------------------------------------
--  Migration Package 
--
---------------------------------------------------------------------------
CREATE OR REPLACE PACKAGE Ordplsgwymigrate AUTHID CURRENT_USER AS

---------------------------------------------------------------------------
--  External procedure declarations
---------------------------------------------------------------------------
PROCEDURE migrate_authed_dads;

END Ordplsgwymigrate;
/

CREATE OR REPLACE PACKAGE BODY Ordplsgwymigrate AS

---------------------------------------------------------------------------
--  Type declarations
---------------------------------------------------------------------------
TYPE vc2_array IS TABLE OF VARCHAR2(128) INDEX BY BINARY_INTEGER;
TYPE dad_array IS TABLE OF VARCHAR2(64);
TYPE user_array IS TABLE OF VARCHAR2(30);

-- private package variables
package_name CONSTANT VARCHAR2( 40 ) := 'ORDSYS.OrdPlsGwyCodeWizard';
admin_dad_name CONSTANT VARCHAR2( 10 ) := 'ORDCWADMIN'; 
admin_user_name CONSTANT VARCHAR2( 30 ) := upper('&cwadmin'); 
dad_pkg CONSTANT VARCHAR2(30) := '.OrdPlsGwyAuthedDADs';
LF CONSTANT VARCHAR2( 1 ) := CHR( 10 );

---------------------------------------------------------------------------
--  Name:
--      get_authed_dads
--
--  Description:
--      Read list of authorized DADs from OrdPlsGwyAuthedDADs package.
---------------------------------------------------------------------------
PROCEDURE get_authed_dads( admin_user IN VARCHAR2,
                           out_dad_list OUT vc2_array,
                           out_user_list OUT vc2_array )
IS
  dad_pkg_name VARCHAR2(64) := admin_user || dad_pkg;
  num_dads NUMBER;
  dad_list vc2_array;
  user_list vc2_array;
BEGIN
  BEGIN
    EXECUTE IMMEDIATE 
        'BEGIN ' ||
        '  :num_dads := ' || dad_pkg_name || '.authed_dads.COUNT; ' ||
        'END;'
      USING OUT num_dads;
    FOR i IN 1..num_dads
    LOOP
      EXECUTE IMMEDIATE 
          'BEGIN ' ||
          '  :dad_name := ' || dad_pkg_name || '.authed_dads( :dad_num ); ' ||
          '  :user_name := ' || dad_pkg_name || '.authed_users( :dad_num ); ' ||
          'END;'
        USING OUT dad_list( i ), IN i, OUT user_list( i );
    END LOOP;
  EXCEPTION
    WHEN OTHERS THEN
      null;
  END;
  out_dad_list := dad_list;
  out_user_list := user_list;
END get_authed_dads;


---------------------------------------------------------------------------
--  Name:
--      set_authed_dads
--
--  Description:
--      Create OrdPlsGwyDADList package with list of authorized DADs.
---------------------------------------------------------------------------
PROCEDURE set_authed_dads( dad_list IN vc2_array,
                           user_list IN vc2_array )
IS
  dad_pkg_name VARCHAR2(64) := admin_user_name || dad_pkg;
  pkg VARCHAR2( 4000 );
  sep VARCHAR2( 2 ); 
BEGIN
  --
  -- First create the package source
  --
  pkg := 'CREATE OR REPLACE PACKAGE ' || dad_pkg_name || ' AS' || LF ||
       '--' || LF ||
       '-- DADs authorized for use with Oracle Multimedia Code Wizard' || LF ||
       '--' || LF ||
       'authed_dads ' || package_name || '.dad_array' || ' := ' || 
                                         package_name || '.dad_array' || '(';
  sep := ' ';
  FOR i IN 1..dad_list.COUNT
  LOOP
    IF dad_list( i ) IS NOT NULL
    THEN
      pkg := pkg || sep || '''' || UPPER( dad_list( i ) ) || '''';
      sep := ', ';
    END IF;
  END LOOP;
  pkg := pkg || ' ); ' || LF;
  pkg := pkg || 'authed_users ' || package_name || '.user_array' || ' := ' || 
                                        package_name || '.user_array' || '(';
  sep := ' ';
  FOR i IN 1..user_list.COUNT
  LOOP
    IF user_list( i ) IS NOT NULL
    THEN
      pkg := pkg || sep || '''' || UPPER( user_list( i ) ) || '''';
      sep := ', ';
    END IF;
  END LOOP;
  pkg := pkg || ' ); ' || LF;
  pkg := pkg || 'END;' || LF;

  --
  -- Now create or replace the package in the database.
  --
  EXECUTE IMMEDIATE pkg;
  
  --
  -- Now grant access to public
  --
  EXECUTE IMMEDIATE 'GRANT EXECUTE ON ' || dad_pkg_name || ' TO PUBLIC';

END set_authed_dads;

---------------------------------------------------------------------------
--  Name:
--      print_dads
--
--  Description:
--      print a dad_list 
---------------------------------------------------------------------------
PROCEDURE print_dads( dad_list IN vc2_array,
                      user_list IN vc2_array )
IS
BEGIN
  IF dad_list IS NULL THEN
    RETURN;
  END IF;

  FOR i IN 1..dad_list.count
  LOOP
    dbms_output.put_line( dad_list(i) || ' / ' || user_list(i) );
  END LOOP;
END print_dads;

---------------------------------------------------------------------------
--  Name:
--      migrate_authed_dads
--
--  Description:
--      Migrate DADS stored in ORDSYS to admin_user_name.
--
---------------------------------------------------------------------------
PROCEDURE migrate_authed_dads
IS
  old_admin_user_name CONSTANT VARCHAR2( 10 ) := 'ORDSYS'; 
  old_dad_pkg_name    CONSTANT VARCHAR2( 40 ) := old_admin_user_name || dad_pkg;
  dad_list vc2_array;
  user_list vc2_array;
BEGIN
  get_authed_dads( old_admin_user_name, dad_list, user_list );

  IF dad_list IS NOT NULL AND dad_list.count > 0 THEN
    dbms_output.put_line( LF || '-----Begin DAD migration---' );
    dbms_output.put_line( LF || 'Now migrating these DAD/USER pairs from ORDSYS to '
                          || admin_user_name );
    print_dads( dad_list, user_list );

    FOR i in 1..dad_list.count
    LOOP
      IF dad_list(i) = admin_dad_name THEN
        user_list(i) := admin_user_name;
      END IF;
    END LOOP;
   
    set_authed_dads( dad_list, user_list );
    get_authed_dads( admin_user_name, dad_list, user_list );

    dbms_output.put_line( LF || 'The current DAD/USER list is: ' );
    print_dads( dad_list, user_list );

    dbms_output.put_line( LF || 'To complete the Code Wizard upgrade, proceed to the' );
    dbms_output.put_line( 'HTTP Server administration page and change the' );
    dbms_output.put_line( 'database username for the ORDCWADMIN DAD from' );
    dbms_output.put_line( 'ORDSYS to ' || upper('&cwadmin') || '.' );
    dbms_output.put_line( LF || 'See the README for more information.' );
    dbms_output.put_line( '-----End DAD migration-----' );
   
    EXECUTE IMMEDIATE 'drop package ' || old_dad_pkg_name;
  ELSE
    dbms_output.put_line( LF || '-----Begin DAD migration---' );
    dbms_output.put_line( 'No DADs were found in the ORDSYS schema.'  );
    dbms_output.put_line( 'No migration was performed.' );
    dbms_output.put_line( '-----End DAD migration-----' );
  END IF;

  EXCEPTION
    WHEN OTHERS THEN
      NULL;
END migrate_authed_dads;

END Ordplsgwymigrate;
/

